<?php

namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use App\Models\TpMapelModel;
use App\Models\MataPelajaranModel;
use App\Models\KelasModel;
use App\Models\UserModel;
use App\Models\TahunAjaranModel;
use App\Models\GuruMapelModel;

class TpMapel extends BaseController
{
    protected $tpMapelModel;
    protected $mataPelajaranModel;
    protected $kelasModel;
    protected $userModel;
    protected $tahunAjaranModel;
    protected $guruMapelModel;

    public function __construct()
    {
        $this->tpMapelModel = new TpMapelModel();
        $this->mataPelajaranModel = new MataPelajaranModel();
        $this->kelasModel = new KelasModel();
        $this->userModel = new UserModel();
        $this->tahunAjaranModel = new TahunAjaranModel();
        $this->guruMapelModel = new GuruMapelModel();
    }

    private function getGuruKelasIds()
    {
        $user_id = session()->get('user_id');
        $user_role = session()->get('role');

        if ($user_role !== 'guru') {
            return [];
        }
        
        $kelas_guru = $this->kelasModel->where('guru_kelas_id', $user_id)->findAll();
        
        $kelas_ids = [];
        foreach ($kelas_guru as $kelas) {
            $kelas_ids[] = $kelas['id_kelas'];
        }
        return $kelas_ids;
    }

    private function getGuruMapelIds()
    {
        $user_id = session()->get('user_id');
        $user_role = session()->get('role');

        if ($user_role !== 'guru') {
            return [];
        }

        $guru_mapel_assignments = $this->guruMapelModel->where('guru_id', $user_id)->findAll();
        
        $mapel_ids = [];
        foreach ($guru_mapel_assignments as $assignment) {
            $mapel_ids[] = $assignment['mapel_id'];
        }
        return $mapel_ids;
    }

    private function getActiveTahunAjaran()
    {
        return $this->tahunAjaranModel->where('aktif', 1)->first();
    }

    private function generateTpCode($kelas_id, $mapel_id, $semester)
    {
        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (!$tahun_ajaran) {
            return 'TP-ERROR';
        }
    
        $tahun_ajaran_id = $tahun_ajaran['id']; // Ditambahkan untuk mencegah Undefined Variable
        
        // Hitung jumlah TP yang sudah ada untuk kombinasi ini
        $count = $this->tpMapelModel
            ->where('kelas_id', $kelas_id)
            ->where('mapel_id', $mapel_id)
            ->where('tahun_ajaran_id', $tahun_ajaran_id)
            ->where('semester', $semester)
            ->countAllResults();
    
        // Format kodenya baru: TP-TA_ID-Semester-KelasID-MapelID-Urutan
        $semester_str = ($semester == 1) ? 'S1' : 'S2';
        return "TP-" . $tahun_ajaran_id . "-" . $semester_str . "-" . $kelas_id . "-" . $mapel_id . "-" . ($count + 1);
    }

    public function index()
    {
        $user_id = session()->get('user_id');
        $guru_mapel_ids = $this->getGuruMapelIds();
        
        $tahun_ajaran = $this->getActiveTahunAjaran();
        
        if (empty($tahun_ajaran)) {
            $data = [
                'title' => 'Manajemen Tujuan Pembelajaran',
                'tp_mapel' => [],
                'kelas_guru' => [],
                'mapel_guru' => [],
                'message' => 'Tidak ada tahun ajaran atau semester aktif yang ditemukan.',
                'user'  => session()->get('user')
            ];
            return view('guru/tp_mapel/index', $data);
        }

        $current_tahun_ajaran_id = $tahun_ajaran['id'];
        
        // --- START MODIFIKASI FILTER SEMESTER ---
        $default_semester = $tahun_ajaran['semester']; // Ambil semester aktif sebagai default
        
        // Tentukan filter semester dari GET (URL), jika tidak ada, gunakan default semester aktif
        $filter_semester = $this->request->getGet('semester') ?? $default_semester; 
        
        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);
        // --- END MODIFIKASI FILTER SEMESTER ---

        if (empty($guru_mapel_ids)) {
            $data = [
                'title' => 'Manajemen Tujuan Pembelajaran',
                'tp_mapel' => [],
                'kelas_guru' => [],
                'mapel_guru' => [],
                'message' => 'Anda belum terdaftar mengampu mata pelajaran apapun.',
                'user'  => session()->get('user')
            ];
            return view('guru/tp_mapel/index', $data);
        }

        $tpMapelQuery = $this->tpMapelModel
                            ->select('tp_mapel.*, mata_pelajaran.nama_mapel, kelas.nama_kelas')
                            ->join('mata_pelajaran', 'mata_pelajaran.id = tp_mapel.mapel_id', 'left')
                            ->join('kelas', 'kelas.id_kelas = tp_mapel.kelas_id', 'left')
                            ->whereIn('tp_mapel.mapel_id', $guru_mapel_ids)
                            // Hanya filter berdasarkan tahun ajaran aktif
                            ->where('tp_mapel.tahun_ajaran_id', $current_tahun_ajaran_id);
                            
        // Terapkan filter semester, KECUALI filter adalah 'all'
        if ($filter_semester !== 'all') {
             // Pastikan nilai filter adalah integer (1 atau 2) sebelum dimasukkan ke where
            $semester_to_filter = (int) $filter_semester;
            $tpMapelQuery->where('tp_mapel.semester', $semester_to_filter);
        }

		$tpMapelQuery->orderBy('tp_mapel.kode_tp', 'ASC');

        if ($is_guru_kelas) {
            $tpMapelQuery->whereIn('tp_mapel.kelas_id', $guru_kelas_ids);
        }

        $tpMapelList = $tpMapelQuery->findAll();
        
        $kelas_info_for_view = [];
        if ($is_guru_kelas) {
             $kelas_info_for_view = $this->kelasModel->whereIn('id_kelas', $guru_kelas_ids)->findAll();
        }

        $mapel_guru = $this->mataPelajaranModel->whereIn('id', $guru_mapel_ids)->findAll();

        $data = [
            'title' => 'Manajemen Tujuan Pembelajaran',
            'tp_mapel' => $tpMapelList,
            'kelas_guru' => $kelas_info_for_view,
            'mapel_guru' => $mapel_guru,
            'filter_semester' => $filter_semester, // BARU: Kirim filter ke view
            'default_semester' => $default_semester, // BARU: Kirim default ke view
            'user'  => session()->get('user')
        ];

        return view('guru/tp_mapel/index', $data);
    }

    public function create()
    {
        $user_id = session()->get('user_id');
        $guru_mapel_ids = $this->getGuruMapelIds();

        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (empty($guru_mapel_ids)) {
            session()->setFlashdata('error', 'Anda belum terdaftar mengampu mata pelajaran apapun.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }

        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (empty($tahun_ajaran)) {
            session()->setFlashdata('error', 'Tidak ada tahun ajaran atau semester aktif yang ditemukan.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }

        $mata_pelajaran_options = $this->mataPelajaranModel->whereIn('id', $guru_mapel_ids)->findAll();

        $kelas_options_for_dropdown = [];
        if ($is_guru_kelas) {
            $kelas_options_for_dropdown = $this->kelasModel->whereIn('id_kelas', $guru_kelas_ids)->findAll();
        } else {
            $kelas_options_for_dropdown = $this->kelasModel->findAll();
        }

        $data = [
            'title' => 'Tambah Tujuan Pembelajaran Baru',
            'mata_pelajaran' => $mata_pelajaran_options,
            'kelas_options' => $kelas_options_for_dropdown,
            'semester_options' => ['Ganjil', 'Genap'],
            'current_semester' => $tahun_ajaran['semester'],
            'validation' => service('validation'),
            'user'  => session()->get('user')
        ];
        return view('guru/tp_mapel/create', $data);
    }

    public function store()
    {
        $user_id = session()->get('user_id');
        $guru_mapel_ids = $this->getGuruMapelIds();

        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        $rules = [
            'mapel_id'       => 'required|numeric',
            'kelas_id'       => 'required|numeric',
            'deskripsi_tp'   => [
                'rules'  => 'required|min_length[10]',
                'errors' => [
                    'required'  => 'Deskripsi Tujuan Pembelajaran wajib diisi.',
                    'min_length' => 'Deskripsi Tujuan Pembelajaran minimal 10 karakter.'
                ]
            ],
            'semester'       => 'required|in_list[Ganjil,Genap]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (empty($tahun_ajaran)) {
            session()->setFlashdata('error', 'Tidak ada tahun ajaran aktif yang ditemukan. Tidak dapat menyimpan.');
            return redirect()->back()->withInput();
        }

        $selected_kelas_id = $this->request->getVar('kelas_id');
        $selected_mapel_id = $this->request->getVar('mapel_id');
        $selected_semester_string = $this->request->getVar('semester');

        $semester_value = ($selected_semester_string == 'Ganjil') ? 1 : (($selected_semester_string == 'Genap') ? 2 : null);

        if ($semester_value === null) {
            session()->setFlashdata('error', 'Nilai semester tidak valid.');
            return redirect()->back()->withInput();
        }

        if (!in_array($selected_mapel_id, $guru_mapel_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk membuat TP untuk mata pelajaran tersebut.');
            return redirect()->back()->withInput();
        }

        if ($is_guru_kelas && !in_array($selected_kelas_id, $guru_kelas_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk membuat TP di kelas tersebut.');
            return redirect()->back()->withInput();
        }
        
        // Memanggil fungsi generateTpCode dan menyimpannya di variabel
        $kode_tp_otomatis = $this->generateTpCode($selected_kelas_id, $selected_mapel_id, $semester_value);

        // Menyimpan data ke database, termasuk kode TP yang dihasilkan secara otomatis
        $this->tpMapelModel->save([
            'mapel_id'         => $selected_mapel_id,
            'kelas_id'         => $selected_kelas_id,
            'kode_tp'          => $kode_tp_otomatis, // Menggunakan kode yang sudah dihasilkan
            'deskripsi_tp'     => $this->request->getVar('deskripsi_tp'),
            'semester'         => $semester_value,
            'tahun_ajaran_id'  => $tahun_ajaran['id'],
        ]);

        session()->setFlashdata('success', 'Tujuan Pembelajaran berhasil ditambahkan.');
        return redirect()->to(base_url('guru/tp_mapel'));
    }

    public function edit($id)
    {
        $user_id = session()->get('user_id');
        $guru_mapel_ids = $this->getGuruMapelIds();

        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (empty($guru_mapel_ids)) {
            session()->setFlashdata('error', 'Anda belum terdaftar mengampu mata pelajaran apapun.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }
        
        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (empty($tahun_ajaran)) {
            session()->setFlashdata('error', 'Tidak ada tahun ajaran atau semester aktif yang ditemukan.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }

        $tp = $this->tpMapelModel->find($id);

        if (empty($tp)) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Tujuan Pembelajaran tidak ditemukan.');
        }

        if (!in_array($tp['mapel_id'], $guru_mapel_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk mengedit TP ini untuk mata pelajaran tersebut.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }

        if ($is_guru_kelas && !in_array($tp['kelas_id'], $guru_kelas_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk mengedit TP ini di kelas tersebut.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }
        
        $mata_pelajaran_options = $this->mataPelajaranModel->whereIn('id', $guru_mapel_ids)->findAll();

        $kelas_options_for_dropdown = [];
        if ($is_guru_kelas) {
            $kelas_options_for_dropdown = $this->kelasModel->whereIn('id_kelas', $guru_kelas_ids)->findAll();
        } else {
            $kelas_options_for_dropdown = $this->kelasModel->findAll();
        }

        $data = [
            'title' => 'Edit Tujuan Pembelajaran',
            'tp' => $tp,
            'mata_pelajaran' => $mata_pelajaran_options,
            'kelas_options' => $kelas_options_for_dropdown,
            'semester_options' => ['Ganjil', 'Genap'],
            'validation' => service('validation'),
            'user'  => session()->get('user')
        ];
        return view('guru/tp_mapel/edit', $data);
    }

    public function update($id)
    {
        $user_id = session()->get('user_id');
        $guru_mapel_ids = $this->getGuruMapelIds();

        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        $rules = [
            'mapel_id'      => 'required|numeric',
            'kelas_id'      => 'required|numeric',
            //'kode_tp'       => 'required|alpha_numeric_punct|min_length[2]|max_length[50]',
            'deskripsi_tp'  => 'required|min_length[10]',
            'semester'      => 'required|in_list[Ganjil,Genap]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }
        
        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (empty($tahun_ajaran)) {
            session()->setFlashdata('error', 'Tidak ada tahun ajaran aktif yang ditemukan. Tidak dapat menyimpan.');
            return redirect()->back()->withInput();
        }

        $selected_kelas_id = $this->request->getVar('kelas_id');
        $selected_mapel_id = $this->request->getVar('mapel_id');
        $selected_semester_string = $this->request->getVar('semester');

        // Mengonversi nilai string semester menjadi nilai integer
        $semester_value = ($selected_semester_string == 'Ganjil') ? 1 : (($selected_semester_string == 'Genap') ? 2 : null);

        if ($semester_value === null) {
            session()->setFlashdata('error', 'Nilai semester tidak valid.');
            return redirect()->back()->withInput();
        }

        if (!in_array($selected_mapel_id, $guru_mapel_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk memperbarui TP untuk mata pelajaran tersebut.');
            return redirect()->back()->withInput();
        }

        if ($is_guru_kelas && !in_array($selected_kelas_id, $guru_kelas_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk memperbarui TP di kelas tersebut.');
            return redirect()->back()->withInput();
        }
        
        $this->tpMapelModel->update($id, [
            'mapel_id'          => $selected_mapel_id,
            'kelas_id'          => $selected_kelas_id,
            //'kode_tp'           => $this->request->getVar('kode_tp'),
            'deskripsi_tp'      => $this->request->getVar('deskripsi_tp'),
            'semester'          => $semester_value,
            'tahun_ajaran_id'   => $tahun_ajaran['id'],
        ]);

        session()->setFlashdata('success', 'Tujuan Pembelajaran berhasil diperbarui.');
        return redirect()->to(base_url('guru/tp_mapel'));
    }

    public function delete($id)
    {
        $user_id = session()->get('user_id');
        $guru_mapel_ids = $this->getGuruMapelIds();

        $guru_kelas_ids = $this->getGuruKelasIds();
        $is_guru_kelas = !empty($guru_kelas_ids);

        if (empty($guru_mapel_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk menghapus TP karena tidak mengampu mata pelajaran apapun.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }

        $tp = $this->tpMapelModel->find($id);

        if (empty($tp)) {
            session()->setFlashdata('error', 'Tujuan Pembelajaran tidak ditemukan.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }

        if (!in_array($tp['mapel_id'], $guru_mapel_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk menghapus TP ini untuk mata pelajaran tersebut.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }

        if ($is_guru_kelas && !in_array($tp['kelas_id'], $guru_kelas_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk menghapus TP ini di kelas tersebut.');
            return redirect()->to(base_url('guru/tp_mapel'));
        }
        
        $this->tpMapelModel->delete($id);
        session()->setFlashdata('success', 'Tujuan Pembelajaran berhasil dihapus.');
        return redirect()->to(base_url('guru/tp_mapel'));
    }
}
