<?php namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use App\Models\KelasModel;
use App\Models\SiswaModel;
use App\Models\TahunAjaranModel;
use App\Models\GuruKokurikulerTemaModel;
use App\Models\GuruKokurikulerDetailModel;
use App\Models\PenilaianKokurikulerModel; // Model Nilai
use App\Models\CatatanWaliKelasModel;

class PenilaianKokurikulerController extends BaseController
{
    protected $kelasModel;
    protected $siswaModel;
    protected $tahunAjaranModel;
    protected $kokurikulerTemaModel;
    protected $kokurikulerDetailModel;
    protected $penilaianKokurikulerModel;
	protected $catatanWaliKelasModel;

    // Rubrik deskripsi (tetap dipertahankan sebagai konteks, meskipun tidak lagi digunakan untuk narasi akhir)
    protected $rubrikDeskripsi = [
        1 => 'masih perlu berlatih dalam', // Kurang
        2 => 'sudah cukup dalam',         // Cukup
        3 => 'sudah baik dalam',          // Baik
        4 => 'sangat baik dalam',         // Sangat Baik
    ];
    
    public function __construct()
    {
        // Inisialisasi semua model yang dibutuhkan
        $this->kelasModel = new KelasModel();
        $this->siswaModel = new SiswaModel();
        $this->tahunAjaranModel = new TahunAjaranModel();
        $this->kokurikulerTemaModel = new GuruKokurikulerTemaModel();
        $this->kokurikulerDetailModel = new GuruKokurikulerDetailModel();
        $this->penilaianKokurikulerModel = new PenilaianKokurikulerModel();
		$this->catatanWaliKelasModel = new CatatanWaliKelasModel();
    }

    /**
     * Menampilkan halaman form untuk generate deskripsi akhir kokurikuler.
     */
    public function index()
    {
        $user_id = session()->get('user_id'); 
        $data['title'] = 'Input Nilai Kokurikuler';
        
        // 1. Ambil Kelas Wali
        $kelas_wali = $this->kelasModel
                            ->select('id_kelas, nama_kelas')
                            ->where('guru_kelas_id', $user_id)
                            ->first();

        if (empty($kelas_wali)) {
            return redirect()->to(base_url('guru/dashboard'))->with('error', 'Anda bukan wali kelas. Fitur ini tidak dapat diakses.');
        }

        $kelas_id = $kelas_wali['id_kelas'];
        $data['kelas_wali'] = $kelas_wali;
        
        // 2. Ambil Tahun Ajaran Aktif dan Semester
        $tahun_ajaran_aktif = $this->tahunAjaranModel->where('aktif', 1)->first();
        $data['tahun_ajaran'] = $tahun_ajaran_aktif;
        
        // 3. Ambil Semua Tema Kokurikuler untuk kelas ini
        $data['list_tema'] = $this->kokurikulerTemaModel
                                ->where('kelas_id', $kelas_id)
                                ->where('tahun_ajaran_id', $tahun_ajaran_aktif['id'])
                                ->findAll();

        return view('guru/penilaian_kokurikuler/index', $data);
    }
    
    // Method untuk menampilkan form input nilai setelah memilih Tema
    public function input($temaId)
    {
        $user_id = session()->get('user_id');
        
        // 1. Ambil data Tema dan Kelas Wali
        $tema = $this->kokurikulerTemaModel->find($temaId);
        $kelas_wali = $this->kelasModel->where('guru_kelas_id', $user_id)->first();
        
        if (empty($tema) || empty($kelas_wali) || $tema['kelas_id'] != $kelas_wali['id_kelas']) {
            return redirect()->to(base_url('guru/penilaian_kokurikuler'))->with('error', 'Tema atau akses kelas tidak valid.');
        }

        // 2. Ambil Siswa di Kelas Wali
        $data['list_siswa'] = $this->siswaModel
                                ->where('kelas_id', $kelas_wali['id_kelas'])
                                ->orderBy('nama_lengkap', 'ASC')
                                ->findAll();

        // 3. Ambil Detail Dimensi yang akan dinilai (Tema 1, 2, 3)
        $data['detail_tema'] = $this->kokurikulerDetailModel
                                ->select('nomor_tema, dimensi, tujuan_pembelajaran')
                                ->where('tema_id', $temaId)
                                ->orderBy('nomor_tema', 'ASC')
                                ->orderBy('id', 'ASC')
                                ->findAll();
                                
        // 4. Ambil Nilai yang sudah tersimpan (jika ada)
        $data['nilai_tersimpan'] = $this->penilaianKokurikulerModel
                                        ->where('tema_id', $temaId)
                                        ->findAll();
                                        
        // Format nilai tersimpan ke dalam array multidimensi agar mudah diakses di view
        $nilai_map = [];
        foreach ($data['nilai_tersimpan'] as $nilai) {
            $key = $nilai['siswa_id'] . '-' . $nilai['nomor_tema'] . '-' . $nilai['dimensi'];
            $nilai_map[$key] = $nilai['nilai'];
        }
        $data['nilai_map'] = $nilai_map;
        
        // 5. Kirim data ke view
        $data['title'] = 'Input Nilai ' . $tema['nama_kegiatan'];
        $data['tema'] = $tema;
        $data['rubrik_deskripsi'] = $this->rubrikDeskripsi;
        
        return view('guru/penilaian_kokurikuler/input', $data);
    }
	
	public function deskripsi()
    {
        $user_id = session()->get('user_id');
        $data['title'] = 'Deskripsi Akhir Kokurikuler';

        // 1. Ambil Kelas Wali
        $kelas_wali = $this->kelasModel
                            ->select('id_kelas, nama_kelas')
                            ->where('guru_kelas_id', $user_id)
                            ->first();

        if (empty($kelas_wali)) {
            return redirect()->to(base_url('guru/dashboard'))->with('error', 'Anda bukan wali kelas. Fitur ini hanya dapat diakses oleh wali kelas.');
        }

        $kelasId = $kelas_wali['id_kelas'];
        
        // 2. Ambil Tahun Ajaran Aktif dan Semester
        $tahun_ajaran = $this->tahunAjaranModel->where('aktif', 1)->first();
        $tahunAjaranId = $tahun_ajaran['id'] ?? null;
        $semester = $tahun_ajaran['semester'] ?? null; 

        if (empty($tahunAjaranId)) {
            return redirect()->to(base_url('guru/dashboard'))->with('error', 'Tahun Ajaran Aktif tidak ditemukan.');
        }
        
        // 3. Ambil daftar siswa di kelas tersebut
        $data['list_siswa'] = $this->siswaModel
                                    ->where('kelas_id', $kelasId)
                                    ->orderBy('nama_lengkap', 'ASC')
                                    ->findAll();

        // 4. Ambil data Tema Kokurikuler yang aktif (Filter TA, Kelas, dan Semester)
        $query = $this->kokurikulerTemaModel
                      ->where('tahun_ajaran_id', $tahunAjaranId)
                      ->where('kelas_id', $kelasId);
                      
        if (!empty($semester) && ($semester == 1 || $semester == 2)) {
            $query->where('semester', $semester);
        }

        $data['tema'] = $query->first();

        return view('guru/penilaian_kokurikuler/deskripsi', $data);
    }

    /**
     * Memproses POST request untuk menghasilkan deskripsi.
     */
		public function generateDeskripsi()
{
    $siswaIdInput = $this->request->getPost('siswa_id');
    $userId = session()->get('user_id');

    // 1. Ambil Data Dasar Guru/Kelas/TA
    $kelasWali = $this->kelasModel->where('guru_kelas_id', $userId)->first();
    $tahunAjaran = $this->tahunAjaranModel->where('aktif', 1)->first();

    if (empty($kelasWali) || empty($tahunAjaran) || !$siswaIdInput) {
        return redirect()->back()->with('error', 'Data guru/siswa/tahun ajaran tidak valid.');
    }

    $kelasId = $kelasWali['id_kelas'];
    $tahunAjaranId = $tahunAjaran['id'];
    $semester = $tahunAjaran['semester'] ?? null;

    // Tentukan daftar siswa yang akan diproses
    $siswaToProcess = [];
    $isBatch = false;

    if ($siswaIdInput === 'all') {
        // Mode Massal: Ambil semua siswa di kelas ini
        $siswaToProcess = $this->siswaModel
                                ->where('kelas_id', $kelasId)
                                ->orderBy('nama_lengkap', 'ASC')
                                ->findAll();
        $isBatch = true;
    } else {
        // Mode Tunggal: Ambil data siswa yang dipilih
        $siswa = $this->siswaModel->find($siswaIdInput);
        if ($siswa) {
            $siswaToProcess[] = $siswa;
        }
    }

    if (empty($siswaToProcess)) {
        return redirect()->back()->with('error', 'Tidak ada siswa yang ditemukan untuk diproses.');
    }

    $successCount = 0;
    $lastProcessedData = null;
    $allResults = []; // <<< ARRAY BARU UNTUK MENYIMPAN SEMUA HASIL

    // 2. Loop dan Proses Setiap Siswa
    foreach ($siswaToProcess as $siswa) {
        $result = $this->_processSingleStudentDeskripsi($siswa['id'], $kelasId, $tahunAjaranId, $semester);

        // Kumpulkan hasil ke dalam array $allResults
        $allResults[] = [
            'nama'            => $siswa['nama_lengkap'],
            'status'          => $result['status'],
            'message'         => $result['message'], // Pesan sukses atau error
            'deskripsi_akhir' => $result['data']['deskripsi_akhir'] ?? null, // Teks deskripsi
        ];

        if ($result['status'] === 'success') {
            $successCount++;
            $lastProcessedData = $result['data'];
        }
    }

    // 3. Kembalikan Hasil
    if ($isBatch) {
        // Mode Massal: Kirim semua hasil melalui flashdata 'batch_results'
        $totalProcessed = count($siswaToProcess);
        return redirect()->to(base_url('guru/penilaian_kokurikuler/deskripsi'))
                         ->with('success', "Berhasil menyimpan deskripsi Kokurikuler untuk {$successCount} dari {$totalProcessed} siswa.")
                         ->with('batch_results', $allResults); // <<< DIKIRIM KE VIEW
    } else {
        // Mode Tunggal: Mengembalikan data tunggal (seperti sebelumnya)
        if ($lastProcessedData) {
            return redirect()->to(base_url('guru/penilaian_kokurikuler/deskripsi'))->with('deskripsi_data', $lastProcessedData);
        }
        return redirect()->back()->with('error', $result['message'] ?? 'Gagal memproses deskripsi untuk siswa ini.');
    }
}
	private function _processSingleStudentDeskripsi(int $siswaId, int $kelasId, int $tahunAjaranId, ?int $semester) : array
	{
		$siswa = $this->siswaModel->find($siswaId);
		if (!$siswa) {
			return ['status' => 'error', 'message' => 'Data siswa tidak ditemukan.'];
		}

		// Ambil Tema Aktif
		$query = $this->kokurikulerTemaModel
					  ->where('kelas_id', $kelasId)
					  ->where('tahun_ajaran_id', $tahunAjaranId);

		if (!empty($semester) && ($semester == 1 || $semester == 2)) {
			$query->where('semester', $semester);
		}
		$tema = $query->first();

		if (empty($tema)) {
			 return ['status' => 'error', 'message' => 'Tema Kokurikuler belum diatur untuk kelas ini.'];
		}

		$temaId = $tema['id'];

		// 2. Ambil Nilai Siswa (Semua Dimensi)
		$nilaiList = $this->penilaianKokurikulerModel
							->select('dimensi, nilai, deskripsi')
							->where(['siswa_id' => $siswaId, 'tema_id' => $temaId])
							->findAll();

		if (empty($nilaiList)) {
			return ['status' => 'error', 'message' => "Siswa {$siswa['nama_lengkap']} belum memiliki nilai Kokurikuler pada tema ini."];
		}

		// 3. Proses Nilai untuk mendapatkan MAX dan MIN skor beserta deskripsinya
		$dimensiData = []; 
		foreach ($nilaiList as $nilai) {
			$dimensi = $nilai['dimensi'];
			$skor = (int)$nilai['nilai'];
			$deskripsi = $nilai['deskripsi'];

			if (!isset($dimensiData[$dimensi]) || $skor > $dimensiData[$dimensi]['skor']) {
				 $dimensiData[$dimensi] = [
					 'skor' => $skor,
					 'deskripsi' => $deskripsi,
					 'dimensi' => $dimensi
				 ];
			}
		}

		$skorList = array_column($dimensiData, 'skor');
		$maxScore = empty($skorList) ? 0 : max($skorList);
		$minScore = empty($skorList) ? 0 : min($skorList);

		// 4. Cari data deskripsi Max dan Min
		$dataMax = null;
		$dataMin = null;

		foreach ($dimensiData as $data) {
			if ($data['skor'] === $maxScore) {
				$dataMax = $data;
				break; 
			}
		}

		if ($maxScore !== $minScore) {
			foreach ($dimensiData as $data) {
				if ($data['skor'] === $minScore) {
					$dataMin = $data;
					break;
				}
			}
		}

		if (empty($dataMax)) {
			return ['status' => 'error', 'message' => "Gagal menemukan deskripsi nilai tertinggi untuk {$siswa['nama_lengkap']}."];
		}

		// 5. Susun Kalimat Akhir (Logika sama seperti sebelumnya)
		$namaPanggilan = esc($siswa['nama_panggilan'] ?? $siswa['nama_lengkap']);
		$temaNames = [];
		if (!empty($tema['tema_1_nama'])) $temaNames[] = $tema['tema_1_nama'];
		if (!empty($tema['tema_2_nama'])) $temaNames[] = $tema['tema_2_nama'];
		if (!empty($tema['tema_3_nama'])) $temaNames[] = $tema['tema_3_nama'];
		//$temaString = implode(', ', $temaNames);
		if (count($temaNames) > 1) {
		// Keluarkan elemen terakhir dan simpan di variabel $lastTema
		$lastTema = array_pop($temaNames); 
		// Gabungkan elemen yang tersisa dengan koma
		$temaString = implode(', ', $temaNames) . ', dan ' . $lastTema;
		} elseif (count($temaNames) === 1) {
		$temaString = $temaNames[0];
		} else {
		$temaString = 'N/A';
		}
		
		$deskripsiAkhir = "Ananda {$namaPanggilan} ";
		$deskripsiAkhir .= "" . esc($dataMax['deskripsi']) . "";

		if ($dataMin) {
			$deskripsiAkhir .= " serta " . esc($dataMin['deskripsi']) . "";
		}

		$deskripsiAkhir .= " dalam tema " . esc($temaString) . ".";

		// 6. SIMPAN DESKRIPSI KE TABEL catatan_wali_kelas
		$existingNote = $this->catatanWaliKelasModel->where([
			'siswa_id'        => $siswaId,
			'semester'        => $semester,
			'tahun_ajaran_id' => $tahunAjaranId,
		])->first();

		if ($existingNote) {
			$this->catatanWaliKelasModel->update($existingNote['id'], ['deskripsi_kokurikuler' => $deskripsiAkhir]);
		} else {
			$dataToSave = [
				'siswa_id'              => $siswaId,
				'semester'              => $semester,
				'tahun_ajaran_id'       => $tahunAjaranId,
				'deskripsi_kokurikuler' => $deskripsiAkhir, 
			];
			$this->catatanWaliKelasModel->insert($dataToSave);
		}

		return [
			'status' => 'success', 
			'message' => 'Deskripsi berhasil disimpan.',
			'data' => [
				'deskripsi_akhir' => $deskripsiAkhir,
				'siswa_id' => $siswaId,
				'max_dimensi' => $dataMax['dimensi'],
				'max_skor' => $maxScore,
				'min_dimensi' => $dataMin ? $dataMin['dimensi'] : null,
				'min_skor' => $minScore,
				'siswa_nama' => $siswa['nama_lengkap'] ?? 'N/A'
			]
		];
	}
    public function store()
    {
        $temaId = $this->request->getPost('tema_id');
        $nilaiData = $this->request->getPost('nilai'); // Array nilai [siswa_id][nomor_tema][dimensi] => nilai
        $tema = $this->kokurikulerTemaModel->find($temaId);
        
        if (!$tema || empty($nilaiData)) {
            return redirect()->back()->with('error', 'Data tidak lengkap atau tema tidak ditemukan.');
        }

        $tahunAjaranId = $tema['tahun_ajaran_id'];
        $kelasId = $tema['kelas_id'];
        
        // Mulai transaksi untuk memastikan data konsisten
        $this->penilaianKokurikulerModel->db->transBegin();
        
        try {
            // 1. Hapus nilai lama (untuk memudahkan proses update/overwrite)
            $this->penilaianKokurikulerModel->where('tema_id', $temaId)->delete();

            // 2. Simpan nilai baru
            foreach ($nilaiData as $siswaId => $nilaiTema) {
                foreach ($nilaiTema as $nomorTema => $nilaiDimensi) {
                    foreach ($nilaiDimensi as $dimensi => $nilai) {
                        // Hilangkan potensi karakter tidak aman dari dimensi
                        $dimensiClean = str_replace(['[', ']'], '', $dimensi);
                        
                        $nilai = (int)$nilai;
                        if ($nilai >= 1 && $nilai <= 4) {
                            $deskripsiRubrik = $this->rubrikDeskripsi[$nilai];
                            
                            $detail = $this->kokurikulerDetailModel
                                         ->select('tujuan_pembelajaran')
                                         ->where('tema_id', $temaId)
                                         ->where('nomor_tema', $nomorTema)
                                         ->where('dimensi', $dimensiClean)
                                         ->first();
                            
                            $tujuan = $detail['tujuan_pembelajaran'] ?? '';

                            $dataToSave = [
                                'tema_id'         => $temaId,
                                'siswa_id'        => $siswaId,
                                'tahun_ajaran_id' => $tahunAjaranId,
                                'kelas_id'        => $kelasId,
                                'nomor_tema'      => $nomorTema,
                                'dimensi'         => $dimensiClean,
                                'nilai'           => $nilai,
                                // Gabungkan rubrik dan tujuan pembelajaran
                                'deskripsi'       => $deskripsiRubrik . ' ' . $tujuan, 
                            ];
                            $this->penilaianKokurikulerModel->insert($dataToSave);
                        }
                    }
                }
            }

            $this->penilaianKokurikulerModel->db->transCommit();
            return redirect()->to(base_url('guru/penilaian_kokurikuler'))->with('success', 'Nilai Kokurikuler berhasil disimpan!');
            
        } catch (\Exception $e) {
            $this->penilaianKokurikulerModel->db->transRollback();
            log_message('error', 'Gagal menyimpan nilai Kokurikuler: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Gagal menyimpan nilai. Silakan coba lagi.');
        }
    }
}