<?php

namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use App\Models\SiswaModel;
use App\Models\KelasModel; 
use App\Models\UserModel; 
use App\Models\KenaikanKelasModel; 
use App\Models\TahunAjaranModel; 

class Siswa extends BaseController
{
    protected $siswaModel;
    protected $kelasModel;
    protected $userModel;
    protected $kenaikanKelasModel; 
    protected $tahunAjaranModel; 

    public function __construct()
    {
        $this->siswaModel = new SiswaModel();
        $this->kelasModel = new KelasModel();
        $this->userModel = new UserModel();
        $this->kenaikanKelasModel = new KenaikanKelasModel(); 
        $this->tahunAjaranModel = new TahunAjaranModel(); 
    }

    public function index()
    {
        // 1. Dapatkan ID guru yang sedang login
        $guruId = session()->get('user_id'); 

        // 2. Cari kelas yang diajar oleh guru ini
        $kelasGuru = $this->kelasModel->where('guru_kelas_id', $guruId)->findAll();

        // Jika guru tidak mengajar kelas mana pun
        if (empty($kelasGuru)) {
            $data = [
                'title' => 'Siswa Kelas Saya',
                'siswa' => [],
                'kelas' => [],
                'message' => 'Anda belum ditugaskan ke kelas mana pun.'
            ];
            return view('guru/siswa/index', $data);
        }

        // Ambil ID kelas dari kelas yang diajar guru
        $kelasIds = array_column($kelasGuru, 'id_kelas'); 

        // 3. Ambil semua siswa yang berada di kelas-kelas tersebut
        $siswaList = $this->siswaModel
                            ->select('siswa.*, kelas.nama_kelas')
                            ->join('kelas', 'kelas.id_kelas = siswa.kelas_id')
                            ->whereIn('siswa.kelas_id', $kelasIds)
                            ->findAll();

        $data = [
            'title' => 'Siswa Kelas Saya',
            'siswa' => $siswaList,
            'kelas' => $kelasGuru, 
            'user'  => session()->get('user')
        ];

        return view('guru/siswa/index', $data);
    }
    
    // ====================================================================
    // FITUR NAIK KELAS BERTAHAP
    // ====================================================================

    public function naikKelas()
    {
        $guruId = session()->get('user_id');
        
        // 1. Ambil Tahun Ajaran Aktif
        $tahunAjaranAktif = $this->tahunAjaranModel->where('aktif', 1)->first();

        // 2. [PENGECEKAN KRITIS] Batasi hanya untuk Semester 2
        if (!$tahunAjaranAktif || (isset($tahunAjaranAktif['semester']) && $tahunAjaranAktif['semester'] != '2')) {
            $currentSemester = $tahunAjaranAktif['semester'] ?? 'Tidak Ditemukan';
            session()->setFlashdata('error', 'Fitur Manajemen Kenaikan Kelas hanya dapat diakses pada **Semester 2** (saat akhir tahun ajaran). Semester saat ini: **' . $currentSemester . '**');
            return redirect()->to(base_url('guru/dashboard')); // Arahkan kembali ke dashboard guru
        }
        
        // 3. Lanjutkan jika sudah Semester 2
        
        $kelasGuru = $this->kelasModel->where('guru_kelas_id', $guruId)->findAll();
        
        $tahunAjaranTujuan = null;
        if ($tahunAjaranAktif) {
            // Ambil Tahun Ajaran berikutnya (misal: ID berikutnya)
            $tahunAjaranTujuan = $this->tahunAjaranModel
                                      ->where('id >', $tahunAjaranAktif['id'])
                                      ->orderBy('id', 'ASC')
                                      ->first();
        }

        $dataKelas = [];
        if ($kelasGuru && $tahunAjaranAktif) {
            foreach ($kelasGuru as $kelas) {
                $kelasId = $kelas['id_kelas'];

                $statusKenaikan = $this->kenaikanKelasModel
                    ->where('kelas_lama_id', $kelasId)
                    ->where('tahun_ajaran_id', $tahunAjaranAktif['id'])
                    ->first(); 

                $isSetRapor = $statusKenaikan ? true : false;
                $isProcessed = $statusKenaikan && $statusKenaikan['is_processed'] == 1;

                $dataKelas[] = [
                    'id_kelas' => $kelasId,
                    'nama_kelas' => $kelas['nama_kelas'],
                    'tahun_ajaran_id' => $tahunAjaranAktif['id'],
                    'tahun_ajaran_nama' => $tahunAjaranAktif['nama_tahun'],
                    'is_set_rapor' => $isSetRapor, 
                    'is_processed' => $isProcessed,
                ];
            }
        }

        $data = [
            'title' => 'Manajemen Kenaikan Kelas',
            'data_kelas' => $dataKelas,
            'ta_aktif' => $tahunAjaranAktif,
            'ta_tujuan' => $tahunAjaranTujuan,
        ];

        return view('guru/siswa/naik_kelas_dashboard', $data);
    }

    // Tahap 1: Set Status Rapor (Default: Naik)
    public function setKenaikanRapor()
    {
        $kelasId = $this->request->getPost('kelas_id');
        $taId = $this->request->getPost('tahun_ajaran_id');
        
        if (!$kelasId || !$taId) {
            session()->setFlashdata('error', 'Kelas dan Tahun Ajaran tidak valid.');
            return redirect()->back();
        }

        $db = \Config\Database::connect();
        $db->transBegin();

        try {
            $sudahDiproses = $this->kenaikanKelasModel
                                ->where(['kelas_lama_id' => $kelasId, 'tahun_ajaran_id' => $taId])
                                ->first();
            if ($sudahDiproses) {
                session()->setFlashdata('warning', 'Status kenaikan rapor untuk kelas ini sudah diset sebelumnya.');
                return redirect()->back();
            }

            $siswaList = $this->siswaModel->where('kelas_id', $kelasId)->findAll();
            $kelasMaksimal = $this->kelasModel->selectMax('id_kelas')->first();
            $idKelasMaksimal = $kelasMaksimal['id_kelas'];
            $kelasSaatIni = $this->kelasModel->find($kelasId);
            
            $dataInsert = [];
            $kelasBaruId = null;

            $kelasBaru = $this->kelasModel->where('id_kelas >', $kelasId)
                                          ->orderBy('id_kelas', 'ASC')
                                          ->first();
            
            if ($kelasBaru) {
                $kelasBaruId = $kelasBaru['id_kelas'];
            }

            foreach ($siswaList as $siswa) {
                $status = 'Naik';
                if ((int)$kelasId == (int)$idKelasMaksimal) {
                    $status = 'Lulus';
                    $kelasBaruId = null; 
                } elseif (!$kelasBaru && (int)$kelasId < (int)$idKelasMaksimal) {
                    $status = 'Tinggal';
                    $kelasBaruId = (int)$kelasId;
                }

                $dataInsert[] = [
                    'siswa_id' => $siswa['id'],
                    'kelas_lama_id' => $kelasId,
                    'kelas_baru_id' => $kelasBaruId,
                    'tahun_ajaran_id' => $taId,
                    'status_kenaikan' => $status,
                    'is_processed' => 0, 
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s'),
                ];
            }

            if (!empty($dataInsert)) {
                $db->table('kenaikan_kelas')->insertBatch($dataInsert); 
            } else {
                 throw new \Exception("Tidak ada siswa yang ditemukan di kelas {$kelasSaatIni['nama_kelas']}.");
            }
            
            if ($db->transStatus() === false) {
                $db->transRollback();
                $lastError = $db->error();
                $errorMessage = "Transaksi Gagal. DB Error: " . (isset($lastError['message']) ? $lastError['message'] : 'Kesalahan tidak diketahui.');
                session()->setFlashdata('error', 'Gagal memproses status kenaikan rapor: ' . $errorMessage);
            } else {
                $db->transCommit();
                session()->setFlashdata('success', 'Status kenaikan **Rapor** untuk siswa di kelas ' . $kelasSaatIni['nama_kelas'] . ' berhasil diset. Tombol Pindah Kelas sekarang aktif. (Status: ' . count($dataInsert) . ' Siswa diproses)');
            }

        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', 'Gagal setKenaikanRapor: ' . $e->getMessage());
            session()->setFlashdata('error', 'Kesalahan Database: Data tidak tersimpan. Error: ' . $e->getMessage());
        }

        return redirect()->to(base_url('guru/siswa/naikKelas'));
    }

    // Tahap 1.5: Menampilkan halaman koreksi status
    public function koreksiKenaikanKelas($kelasLamaId, $taId)
    {
        // 1. Ambil data kenaikan yang BELUM diproses di kelas ini
        $siswaKenaikanData = $this->kenaikanKelasModel
            ->select('kenaikan_kelas.*, siswa.nama_lengkap, kelas_lama.nama_kelas as nama_kelas_lama, kelas_baru.nama_kelas as nama_kelas_baru')
            ->join('siswa', 'siswa.id = kenaikan_kelas.siswa_id')
            ->join('kelas as kelas_lama', 'kelas_lama.id_kelas = kenaikan_kelas.kelas_lama_id')
            ->join('kelas as kelas_baru', 'kelas_baru.id_kelas = kenaikan_kelas.kelas_baru_id', 'left')
            ->where('kenaikan_kelas.kelas_lama_id', $kelasLamaId)
            ->where('kenaikan_kelas.tahun_ajaran_id', $taId)
            ->where('kenaikan_kelas.is_processed', 0)
            ->findAll();

        if (empty($siswaKenaikanData)) {
            session()->setFlashdata('error', 'Semua siswa di kelas ini sudah dipindah atau status kenaikan belum diset.');
            return redirect()->to(base_url('guru/siswa/naikKelas'));
        }

        $data = [
            'title' => 'Koreksi Status Kenaikan Kelas',
            'siswa_data' => $siswaKenaikanData,
            'kelas_lama_id' => $kelasLamaId,
            'ta_id' => $taId,
            'status_options' => ['Naik', 'Tinggal', 'Lulus'] 
        ];

        // Memanggil view baru
        return view('guru/siswa/koreksi_status_kenaikan', $data);
    }
    
    public function prosesPindahKelas()
    {
        // 1. Pastikan nama ini cocok dengan nama di input hidden (Langkah 1)
        $kelasLamaId = $this->request->getPost('kelas_lama_id'); 
        $taIdLama = $this->request->getPost('ta_id_lama');       
        $taIdBaru = $this->request->getPost('ta_id_baru');       

        if (!$kelasLamaId || !$taIdLama || !$taIdBaru) {
            session()->setFlashdata('error', 'Parameter pindah kelas tidak lengkap.');
            return redirect()->to(base_url('guru/siswa/naikKelas'));
        }

        $db = \Config\Database::connect();
        $db->transBegin();
        $totalSiswaDiproses = 0;

        try {
            // 2. Ambil data siswa yang statusnya sudah final (Naik/Tinggal/Lulus) di kelas dan TA lama ini.
            $siswaFinal = $this->kenaikanKelasModel
                ->where('kelas_lama_id', $kelasLamaId)
                ->where('tahun_ajaran_id', $taIdLama)
                ->whereIn('status_kenaikan', ['Naik', 'Tinggal', 'Lulus'])
                ->findAll();

            if (empty($siswaFinal)) {
                throw new \Exception("Tidak ada siswa dengan status final (Naik/Tinggal/Lulus) yang ditemukan untuk kelas ini.");
            }

            foreach ($siswaFinal as $dataKenaikan) {
                $siswaId = $dataKenaikan['siswa_id'];
                $status = $dataKenaikan['status_kenaikan'];
                $kelasBaruId = $dataKenaikan['kelas_baru_id']; // Sudah dihitung saat koreksi atau saat set status awal

                // 3. Update Tabel Siswa (Memindah siswa ke kelas dan TA baru)
                
                $siswaUpdateData = [
                    'kelas_id' => $kelasBaruId, // Null jika Lulus
                    'tahun_ajaran_id' => $taIdBaru, // Pindah ke TA baru
                    'is_aktif' => ($status == 'Lulus' ? 0 : 1), // Nonaktif jika Lulus
                    'updated_at' => date('Y-m-d H:i:s')
                ];

                // Update data siswa
                $updatedSiswa = $this->siswaModel->update($siswaId, $siswaUpdateData);

                if (!$updatedSiswa) {
                    throw new \Exception("Gagal update data siswa (ID: $siswaId).");
                }

                // 4. Set is_processed = 1 di tabel kenaikan_kelas
                $updatedKenaikan = $this->kenaikanKelasModel->update($dataKenaikan['id'], ['is_processed' => 1]);
                
                if (!$updatedKenaikan) {
                    throw new \Exception("Gagal set is_processed (ID: {$dataKenaikan['id']}).");
                }
                
                $totalSiswaDiproses++;
            }
            
            // 5. Commit Transaksi
            if ($db->transStatus() === false) {
                $db->transRollback();
                session()->setFlashdata('error', 'Gagal pindah kelas: Transaksi Database gagal.');
            } else {
                $db->transCommit();
                session()->setFlashdata('success', "Proses Pindah Kelas Selesai! $totalSiswaDiproses siswa di kelas ini berhasil dipindahkan ke Tahun Ajaran Baru.");
            }

        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', 'Gagal prosesPindahKelas: ' . $e->getMessage());
            session()->setFlashdata('error', 'Kesalahan Pindah Kelas: ' . $e->getMessage());
        }

        // Kembali ke dashboard kenaikan kelas
        return redirect()->to(base_url('guru/siswa/naikKelas'));
    }
	
	public function saveKoreksiStatusBatch()
    {
        $statusArray = $this->request->getPost('status');
        $kenaikanIdArray = $this->request->getPost('kenaikan_id');
        $kelasLamaId = $this->request->getPost('kelas_lama_id');
        $taId = $this->request->getPost('ta_id');
        
        if (empty($statusArray) || empty($kenaikanIdArray)) {
            session()->setFlashdata('error', 'Tidak ada data siswa untuk disimpan.');
            return redirect()->back();
        }

        $db = \Config\Database::connect();
        $db->transBegin();
        $updatesCount = 0;

        try {
            $kelasMaksimal = $this->kelasModel->selectMax('id_kelas')->first();
            $idKelasMaksimal = $kelasMaksimal['id_kelas'];

            foreach ($statusArray as $siswaId => $newStatus) {
                $kenaikanId = $kenaikanIdArray[$siswaId];
                
                // 1. Tentukan kelas tujuan (kelas_baru_id)
                $kelasBaruId = null; 

                if ($newStatus == 'Tinggal') {
                    $kelasBaruId = $kelasLamaId; 
                } elseif ($newStatus == 'Lulus') {
                    $kelasBaruId = null; 
                } else { // Status: Naik
                    $kelasBaru = $this->kelasModel->where('id_kelas >', $kelasLamaId)
                                                  ->orderBy('id_kelas', 'ASC')
                                                  ->first();
                    $kelasBaruId = $kelasBaru ? $kelasBaru['id_kelas'] : null;
                    
                    if ((int)$kelasLamaId == (int)$idKelasMaksimal) {
                         // Hanya untuk keamanan
                         $newStatus = 'Lulus';
                         $kelasBaruId = null;
                    }
                }

                // 2. Data yang akan diupdate
                $updateData = [
                    'status_kenaikan' => $newStatus,
                    'kelas_baru_id' => $kelasBaruId,
                    'updated_at' => date('Y-m-d H:i:s')
                ];

                // 3. Update dan cek kegagalan validasi
                $updated = $this->kenaikanKelasModel->update($kenaikanId, $updateData);
                
                if (!$updated) {
                    if ($this->kenaikanKelasModel->errors()) {
                        $errors = implode('; ', $this->kenaikanKelasModel->errors());
                        throw new \Exception("Gagal Validasi Model untuk Siswa ID {$siswaId}: " . $errors);
                    }
                    throw new \Exception("Update data kenaikan kelas gagal untuk Siswa ID {$siswaId}.");
                }
                $updatesCount++;
            }
            
            // 4. Commit Transaksi
            if ($db->transStatus() === false) {
                $db->transRollback();
                session()->setFlashdata('error', 'Gagal update status: Transaksi Gagal pada DB.');
            } else {
                $db->transCommit();
                session()->setFlashdata('success', "$updatesCount status siswa berhasil dikoreksi dan disimpan.");
            }

        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', 'Gagal saveKoreksiStatusBatch: ' . $e->getMessage());
            session()->setFlashdata('error', 'Kesalahan Database: ' . $e->getMessage());
        }

        // Kembali ke halaman koreksi atau dashboard
        return redirect()->to(base_url("guru/siswa/koreksiKenaikanKelas/{$kelasLamaId}/{$taId}"));
    }
}