<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\SiswaModel;
use App\Models\KelasModel;
// Tambahkan use statement untuk PhpSpreadsheet (setelah diinstal)
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Reader\Csv;
use PhpOffice\PhpSpreadsheet\Reader\Xlsx;


class Siswa extends BaseController
{
    protected $siswaModel;
    protected $kelasModel;

    public function __construct()
    {
        $this->siswaModel = new SiswaModel();
        $this->kelasModel = new KelasModel();
    }

    public function index()
    {
        $daftarSiswa = $this->siswaModel->findAll();

        $kelasData = $this->kelasModel->findAll();
        $kelasMap = array_column($kelasData, 'nama_kelas', 'id_kelas');

        foreach ($daftarSiswa as &$siswa) {
            $siswa['nama_kelas'] = $kelasMap[$siswa['kelas_id']] ?? '-';
        }
        unset($siswa);

        $data = [
            'title'      => 'Manajemen Siswa',
            'daftarSiswa' => $daftarSiswa
        ];
        return view('admin/siswa/index', $data);
    }

    public function create()
    {
        $kelasOptions = $this->kelasModel->findAll();

        $data = [
            'title'      => 'Tambah Siswa Baru',
            'validation' => \Config\Services::validation(),
            'kelasOptions' => $kelasOptions
        ];
        return view('admin/siswa/create', $data);
    }

    public function store()
    {
        $rules = [
            'nama_lengkap'          => 'required|min_length[3]',
			'nama_panggilan'        => 'required|min_length[3]',
			'jenis_kelamin'         => 'permit_empty|min_length[3]',
            'agama'                 => 'permit_empty|min_length[3]',
            'nisn'                  => 'permit_empty|exact_length[10]|numeric|is_unique[siswa.nisn]',
            'nis'                   => 'permit_empty|max_length[10]|numeric|is_unique[siswa.nis]',
            'pendidikan_sebelumnya' => 'permit_empty',
            'kelas_id'              => 'required|is_natural_no_zero',
            'tempat_lahir'          => 'permit_empty|min_length[3]',
            'tanggal_lahir'         => 'permit_empty|valid_date',
            'alamat'                => 'permit_empty|min_length[5]',
            'kelurahan_desa'        => 'permit_empty|min_length[3]',
            'kapanewon'             => 'permit_empty|min_length[3]',
            'kabupaten_kota'        => 'permit_empty|min_length[3]',
            'provinsi'              => 'permit_empty|min_length[3]',
            'nama_ayah'             => 'permit_empty|min_length[3]',
            'pekerjaan_ayah'        => 'permit_empty|min_length[3]',
            'nama_ibu'              => 'permit_empty|min_length[3]',
            'pekerjaan_ibu'         => 'permit_empty|min_length[3]',
            'nama_wali'             => 'permit_empty|min_length[3]',
            'pekerjaan_wali'        => 'permit_empty|min_length[3]',
            'alamat_wali'           => 'permit_empty|min_length[5]',
        ];

        $messages = [
            'nama_lengkap' => [
                'required'   => 'Nama Lengkap harus diisi.',
                'min_length' => 'Nama Lengkap minimal 3 karakter.'
            ],
            'nama_panggilan' => [
                'required'   => 'Nama Panggilan harus diisi.',
                'min_length' => 'Nama Panggilan minimal 3 karakter.'
            ],
			'jenis_kelamin' => [
                'min_length' => 'Nama Panggilan minimal 3 karakter.'
            ],
			'nisn' => [
                'exact_length' => 'NISN harus 10 digit.',
                'numeric'      => 'NISN harus berupa angka.',
                'is_unique'    => 'NISN ini sudah terdaftar.'
            ],
            'nis' => [
                'max_length' => 'NIS maksimal 10 digit.',
                'numeric'    => 'NIS harus berupa angka.',
                'is_unique'  => 'NIS ini sudah terdaftar.'
            ],
            'kelas_id' => [
                'required'          => 'Kelas harus dipilih.',
                'is_natural_no_zero'=> 'Kelas tidak valid.'
            ],
            'tanggal_lahir' => [
                'valid_date' => 'Format Tanggal Lahir tidak valid.'
            ],
            'alamat' => [
                'min_length' => 'Alamat minimal 5 karakter.'
            ],
            'alamat_wali' => [
                'min_length' => 'Alamat Wali minimal 5 karakter.'
            ],
        ];

        if (!$this->validate($rules, $messages)) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        $dataToSave = [
            'nisn'                  => $this->request->getVar('nisn'),
            'nis'                   => $this->request->getVar('nis'),
            'pendidikan_sebelumnya' => $this->request->getVar('pendidikan_sebelumnya'),
            'nama_lengkap'          => $this->request->getVar('nama_lengkap'),
			'nama_panggilan'        => $this->request->getVar('nama_panggilan'),
            'jenis_kelamin'         => $this->request->getVar('jenis_kelamin'),
			'agama'                 => $this->request->getVar('agama'),
			'kelas_id'              => $this->request->getVar('kelas_id'),
            'tempat_lahir'          => $this->request->getVar('tempat_lahir'),
            'tanggal_lahir'         => $this->request->getVar('tanggal_lahir'),
            'alamat'                => $this->request->getVar('alamat'),
            'kelurahan_desa'        => $this->request->getVar('kelurahan_desa'),
            'kapanewon'             => $this->request->getVar('kapanewon'),
            'kabupaten_kota'        => $this->request->getVar('kabupaten_kota'),
            'provinsi'              => $this->request->getVar('provinsi'),
            'nama_ayah'             => $this->request->getVar('nama_ayah'),
            'pekerjaan_ayah'        => $this->request->getVar('pekerjaan_ayah'),
            'nama_ibu'              => $this->request->getVar('nama_ibu'),
            'pekerjaan_ibu'         => $this->request->getVar('pekerjaan_ibu'),
            'nama_wali'             => $this->request->getVar('nama_wali'),
            'pekerjaan_wali'        => $this->request->getVar('pekerjaan_wali'),
            'alamat_wali'           => $this->request->getVar('alamat_wali'),
        ];

        $this->siswaModel->save($dataToSave);

        session()->setFlashdata('success', 'Data siswa berhasil ditambahkan.');
        return redirect()->to(base_url('admin/siswa'));
    }

    public function edit($id)
    {
        $siswa = $this->siswaModel->find($id);
        if (empty($siswa)) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Data siswa tidak ditemukan.');
        }

        $kelasOptions = $this->kelasModel->findAll();

        $data = [
            'title'      => 'Edit Data Siswa',
            'validation' => \Config\Services::validation(),
            'siswa'      => $siswa,
            'kelasOptions' => $kelasOptions
        ];
        return view('admin/siswa/edit', $data);
    }

    public function update($id)
    {
        $rules = [
            'nama_lengkap'          => 'required|min_length[3]',
			'nama_panggilan'        => 'required|min_length[3]',
            'jenis_kelamin'         => 'permit_empty|min_length[3]',
			'agama'                 => 'permit_empty|min_length[3]',
			'nisn'                  => "permit_empty|exact_length[10]|numeric|is_unique[siswa.nisn,id,{$id}]",
            'nis'                   => "permit_empty|max_length[10]|numeric|is_unique[siswa.nis,id,{$id}]",
            'pendidikan_sebelumnya' => 'permit_empty',
            'kelas_id'              => 'required|is_natural_no_zero',
            'tempat_lahir'          => 'permit_empty|min_length[3]',
            'tanggal_lahir'         => 'permit_empty|valid_date',
            'alamat'                => 'permit_empty|min_length[5]',
            'kelurahan_desa'        => 'permit_empty|min_length[3]',
            'kapanewon'             => 'permit_empty|min_length[3]',
            'kabupaten_kota'        => 'permit_empty|min_length[3]',
            'provinsi'              => 'permit_empty|min_length[3]',
            'nama_ayah'             => 'permit_empty|min_length[3]',
            'pekerjaan_ayah'        => 'permit_empty|min_length[3]',
            'nama_ibu'              => 'permit_empty|min_length[3]',
            'pekerjaan_ibu'         => 'permit_empty|min_length[3]',
            'nama_wali'             => 'permit_empty|min_length[3]',
            'pekerjaan_wali'        => 'permit_empty|min_length[3]',
            'alamat_wali'           => 'permit_empty|min_length[5]',
        ];

        $messages = [
            'nama_lengkap' => [
                'required'   => 'Nama Lengkap harus diisi.',
                'min_length' => 'Nama Lengkap minimal 3 karakter.'
            ],
            'nama_panggilan' => [
                'required'   => 'Nama Panggilan harus diisi.',
                'min_length' => 'Nama Panggilan minimal 3 karakter.'
            ],
			'jenis_kelamin' => [
                'min_length' => 'Nama Panggilan minimal 3 karakter.'
			],
			'nisn' => [
                'exact_length' => 'NISN harus 10 digit.',
                'numeric'      => 'NISN harus berupa angka.',
                'is_unique'    => 'NISN ini sudah terdaftar.'
            ],
            'nis' => [
                'max_length' => 'NIS maksimal 10 digit.',
                'numeric'    => 'NIS harus berupa angka.',
                'is_unique'  => 'NIS ini sudah terdaftar.'
            ],
            'kelas_id' => [
                'required'          => 'Kelas harus dipilih.',
                'is_natural_no_zero'=> 'Kelas tidak valid.'
            ],
            'tanggal_lahir' => [
                'valid_date' => 'Format Tanggal Lahir tidak valid.'
            ],
            'alamat' => [
                'min_length' => 'Alamat minimal 5 karakter.'
            ],
            'alamat_wali' => [
                'min_length' => 'Alamat Wali minimal 5 karakter.'
            ],
        ];

        if (!$this->validate($rules, $messages)) {
            return redirect()->back()->withInput()->with('validation', $this->validator);
        }

        $dataToUpdate = [
            'nisn'                  => $this->request->getVar('nisn'),
            'nis'                   => $this->request->getVar('nis'),
            'pendidikan_sebelumnya' => $this->request->getVar('pendidikan_sebelumnya'),
            'nama_lengkap'          => $this->request->getVar('nama_lengkap'),
            'nama_panggilan'        => $this->request->getVar('nama_panggilan'),
			'jenis_kelamin'         => $this->request->getVar('jenis_kelamin'),
			'agama'                 => $this->request->getVar('agama'),
			'kelas_id'              => $this->request->getVar('kelas_id'),
            'tempat_lahir'          => $this->request->getVar('tempat_lahir'),
            'tanggal_lahir'         => $this->request->getVar('tanggal_lahir'),
            'alamat'                => $this->request->getVar('alamat'),
            'kelurahan_desa'        => $this->request->getVar('kelurahan_desa'),
            'kapanewon'             => $this->request->getVar('kapanewon'),
            'kabupaten_kota'        => $this->request->getVar('kabupaten_kota'),
            'provinsi'              => $this->request->getVar('provinsi'),
            'nama_ayah'             => $this->request->getVar('nama_ayah'),
            'pekerjaan_ayah'        => $this->request->getVar('pekerjaan_ayah'),
            'nama_ibu'              => $this->request->getVar('nama_ibu'),
            'pekerjaan_ibu'         => $this->request->getVar('pekerjaan_ibu'),
            'nama_wali'             => $this->request->getVar('nama_wali'),
            'pekerjaan_wali'        => $this->request->getVar('pekerjaan_wali'),
            'alamat_wali'           => $this->request->getVar('alamat_wali'),
        ];

        $this->siswaModel->update($id, $dataToUpdate);

        session()->setFlashdata('success', 'Data siswa berhasil diperbarui.');
        return redirect()->to(base_url('admin/siswa'));
    }

    public function delete($id)
    {
        $siswa = $this->siswaModel->find($id);
        if (empty($siswa)) {
            session()->setFlashdata('error', 'Data siswa tidak ditemukan.');
            return redirect()->to(base_url('admin/siswa'));
        }

        $this->siswaModel->delete($id);

        session()->setFlashdata('success', 'Data siswa berhasil dihapus.');
        return redirect()->to(base_url('admin/siswa'));
    }

    public function importView()
    {
        $data = [
            'title'      => 'Import Data Siswa',
            'validation' => \Config\Services::validation(),
        ];
        return view('admin/siswa/import', $data);
    }

    public function processImport()
{
    // Aturan validasi untuk file yang diunggah
    $rules = [
        'file_excel' => 'uploaded[file_excel]|max_size[file_excel,2048]|ext_in[file_excel,xls,xlsx,csv]',
    ];
    $messages = [
        'file_excel' => [
            'uploaded' => 'Anda harus memilih file untuk diupload.',
            'max_size' => 'Ukuran file terlalu besar (maks 2MB).',
            'ext_in'   => 'Format file tidak didukung. Gunakan .xls, .xlsx, atau .csv.',
        ],
    ];

    if (!$this->validate($rules, $messages)) {
        // Jika validasi file awal gagal, pesan error akan ditampilkan di view secara otomatis
         dd($this->validator->getErrors()); // <<< TAMBAHKAN INI SEMENTARA untuk melihat error validasi file
		return redirect()->back()->withInput()->with('validation', $this->validator);
    }

    $file = $this->request->getFile('file_excel');
    $fileName = $file->getName();
    $fileExtension = $file->getClientExtension();

    // Pindahkan file ke direktori writable
    $newName = $file->getRandomName();
    $file->move(WRITEPATH . 'uploads', $newName);
    $filePath = WRITEPATH . 'uploads/' . $newName;

    // --- DEBUGGING POINT 1: Pastikan file berhasil dipindahkan ---
    if (!file_exists($filePath)) {
        session()->setFlashdata('error', 'Gagal memindahkan file ke direktori uploads. Periksa izin folder.');
        return redirect()->back();
    }
    // echo "File dipindahkan ke: " . $filePath . "<br>";
    // die(); // Hentikan eksekusi di sini untuk melihat path file


    $reader = null;
    if ($fileExtension == 'csv') {
        $reader = new Csv();
    } elseif ($fileExtension == 'xls' || $fileExtension == 'xlsx') {
        $reader = IOFactory::createReaderForFile($filePath);
    } else {
        session()->setFlashdata('error', 'Format file tidak didukung.');
        return redirect()->back();
    }

    try {
        $spreadsheet = $reader->load($filePath);
        $sheet = $spreadsheet->getActiveSheet();
        $dataExcel = $sheet->toArray(null, true, true, true);

        // --- DEBUGGING POINT 2: Periksa apakah data Excel terbaca dengan benar ---
        // echo "<pre>";
        // print_r($dataExcel);
        // die(); // Hentikan eksekusi di sini untuk melihat semua data dari Excel


        // Hapus baris header (asumsi baris pertama adalah header)
        array_shift($dataExcel); // Ini menghapus baris 1 (A, B, C, ...)
        array_shift($dataExcel); // <<< TAMBAHKAN BARIS INI untuk menghapus baris header sebenarnya (NISN, NIS, ...)
		array_shift($dataExcel);
        
		$importedCount = 0;
        $failedCount = 0;
        $errors = [];

        $columnMapping = [
            'nisn'                  => 'A',
            'nis'                   => 'B',
            'pendidikan_sebelumnya' => 'C',
            'nama_lengkap'          => 'D',
            'nama_panggilan'        => 'E',
			'jenis_kelamin'			=> 'F',
			'agama'                 => 'G',
			'kelas_id'              => 'H',
            'tempat_lahir'          => 'I',
            'tanggal_lahir'         => 'J',
            'alamat'                => 'K',
            'kelurahan_desa'        => 'L',
            'kapanewon'             => 'M',
            'kabupaten_kota'        => 'N',
            'provinsi'              => 'O',
            'nama_ayah'             => 'P',
            'pekerjaan_ayah'        => 'Q',
            'nama_ibu'              => 'R',
            'pekerjaan_ibu'         => 'S',
            'nama_wali'             => 'T',
            'pekerjaan_wali'        => 'U',
            'alamat_wali'           => 'V',
        ];

        foreach ($dataExcel as $rowNum => $row) {
            if (empty(array_filter($row))) {
                continue;
            }

            $dataSiswa = [];
            foreach ($columnMapping as $dbColumn => $excelColumn) {
                $dataSiswa[$dbColumn] = trim($row[$excelColumn] ?? '');
            }

            if (empty($dataSiswa['nisn'])) $dataSiswa['nisn'] = null;
            if (empty($dataSiswa['nis'])) $dataSiswa['nis'] = null;

            if (!empty($dataSiswa['tanggal_lahir'])) {
                if (is_numeric($dataSiswa['tanggal_lahir'])) {
                    try {
                        $excelDate = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($dataSiswa['tanggal_lahir']);
                        $dataSiswa['tanggal_lahir'] = $excelDate->format('Y-m-d');
                    } catch (\Exception $e) {
                        $errors[] = "Baris " . ($rowNum) . ": Format Tanggal Lahir tidak valid (Excel Date): " . $dataSiswa['tanggal_lahir'];
                        $failedCount++;
                        continue;
                    }
                } else {
                    try {
                        $dateTime = new \DateTime($dataSiswa['tanggal_lahir']);
                        $dataSiswa['tanggal_lahir'] = $dateTime->format('Y-m-d');
                    } catch (\Exception $e) {
                        $errors[] = "Baris " . ($rowNum) . ": Format Tanggal Lahir tidak valid (String Date). Nilai: " . $dataSiswa['tanggal_lahir'];
                        $failedCount++;
                        continue;
                    }
                }
            } else {
                $dataSiswa['tanggal_lahir'] = null;
            }

            // --- DEBUGGING POINT 3: Periksa data siswa sebelum validasi model ---
            // echo "<h3>Data Siswa Baris " . $rowNum . "</h3>";
            // echo "<pre>";
            // print_r($dataSiswa);
            // die(); // Hentikan di sini untuk setiap baris

            // Cek duplikasi NISN/NIS secara manual sebelum save untuk import
            $existingSiswa = null;
            if (!empty($dataSiswa['nisn'])) {
                $existingSiswa = $this->siswaModel->where('nisn', $dataSiswa['nisn'])->first();
            }
            if (!$existingSiswa && !empty($dataSiswa['nis'])) { // Hanya cek NIS jika NISN tidak ditemukan/kosong
                $existingSiswa = $this->siswaModel->where('nis', $dataSiswa['nis'])->first();
            }

            if ($existingSiswa && (!empty($dataSiswa['nisn']) || !empty($dataSiswa['nis']))) {
                $errors[] = "Baris " . ($rowNum) . ": NISN ('" . ($dataSiswa['nisn'] ?? 'null') . "') atau NIS ('" . ($dataSiswa['nis'] ?? 'null') . "') sudah ada di database. Data tidak diimport.";
                $failedCount++;
                continue;
            }

            if ($this->siswaModel->validate($dataSiswa)) {
                $this->siswaModel->insert($dataSiswa);
                // --- DEBUGGING POINT 4: Periksa jika ada error setelah insert (jika tidak berhasil) ---
                if ($this->siswaModel->errors()) {
                    $errors[] = "Baris " . ($rowNum) . ": Gagal insert ke DB. Error Model: " . implode(", ", $this->siswaModel->errors());
                    $failedCount++;
                } else {
                    $importedCount++;
                }
            } else {
                $errors[] = "Baris " . ($rowNum) . ": Validasi Gagal. " . implode(", ", $this->siswaModel->errors());
                $failedCount++;
            }
        }

        // Hapus file sementara setelah selesai
        unlink($filePath);

        if ($failedCount > 0) {
            session()->setFlashdata('warning', "Import selesai dengan beberapa masalah. Berhasil: {$importedCount}, Gagal: {$failedCount}.");
            session()->setFlashdata('import_errors', $errors);
        } else {
            session()->setFlashdata('success', "Import data siswa berhasil. Total siswa ditambahkan: {$importedCount}.");
        }
        return redirect()->to(base_url('admin/siswa'));

    } catch (\Exception $e) {
        // Hapus file sementara jika ada error saat membaca
        if (file_exists($filePath)) {
            unlink($filePath);
        }
        session()->setFlashdata('error', 'Terjadi kesalahan saat memproses file: ' . $e->getMessage());
        return redirect()->back()->withInput();
    }
}
}