<?php

namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use App\Models\SiswaModel;
use App\Models\KelasModel;
use App\Models\TahunAjaranModel;
use App\Models\EkskulSiswaModel;
use App\Models\EkstrakurikulerMasterModel;

class EkskulWaliKelas extends BaseController
{
    protected $siswaModel;
    protected $kelasModel;
    protected $tahunAjaranModel;
    protected $ekskulSiswaModel;
    protected $ekskulMasterModel;

    public function __construct()
    {
        $this->siswaModel = new SiswaModel();
        $this->kelasModel = new KelasModel();
        $this->tahunAjaranModel = new TahunAjaranModel();
        $this->ekskulSiswaModel = new EkskulSiswaModel();
        $this->ekskulMasterModel = new EkstrakurikulerMasterModel();
    }
    
    // Fungsi bantu untuk mendapatkan ID kelas yang diajar oleh guru yang sedang login (sebagai wali kelas)
    private function getGuruKelasIds()
    {
        $user_id = session()->get('user_id');
        $kelas_guru = $this->kelasModel->where('guru_kelas_id', $user_id)->findAll();
        
        $kelas_ids = [];
        foreach ($kelas_guru as $kelas) {
            $kelas_ids[] = $kelas['id_kelas'];
        }
        return $kelas_ids;
    }

    // Fungsi bantu untuk mendapatkan ID ekskul yang dipegang oleh guru yang sedang login (sebagai penanggung jawab)
    private function getPenanggungJawabEkskulIds()
    {
        $user_id = session()->get('user_id');
        $ekskul_master = $this->ekskulMasterModel->where('penanggung_jawab_id', $user_id)->findAll();

        $ekskul_ids = [];
        foreach ($ekskul_master as $ekskul) {
            $ekskul_ids[] = $ekskul['id'];
        }
        return $ekskul_ids;
    }

    private function getActiveTahunAjaran()
    {
        return $this->tahunAjaranModel->where('aktif', 1)->first();
    }
    
    public function index()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $ekskul_penanggung_jawab_ids = $this->getPenanggungJawabEkskulIds();

        $is_guru_kelas = !empty($guru_kelas_ids);
        $is_penanggung_jawab_ekskul = !empty($ekskul_penanggung_jawab_ids);

        // Jika user bukan wali kelas atau penanggung jawab ekskul, tampilkan pesan error
        if (!$is_guru_kelas && !$is_penanggung_jawab_ekskul) {
            $data = [
                'title' => 'Input Nilai Ekstrakurikuler',
                'ekskul_list' => [],
                'siswa_data' => [],
                'message' => 'Anda tidak terdaftar sebagai wali kelas atau penanggung jawab ekstrakurikuler. Halaman ini hanya bisa diakses oleh kedua peran tersebut.'
            ];
            return view('guru/ekskul_wali_kelas/index', $data);
        }

        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (empty($tahun_ajaran)) {
             $data = [
                'title' => 'Input Nilai Ekstrakurikuler',
                'siswa_data' => [],
                'siswa_ekskul' => [],
                'message' => 'Tidak ada tahun ajaran aktif yang ditemukan.'
            ];
            return view('guru/ekskul_wali_kelas/index', $data);
        }
        
        $builder = $this->ekskulSiswaModel
            ->select('ekskul_siswa.*, siswa.nama_lengkap, kelas.nama_kelas, ekstrakurikuler_master.nama_ekskul')
            ->join('siswa', 'siswa.id = ekskul_siswa.siswa_id')
            ->join('kelas', 'kelas.id_kelas = siswa.kelas_id')
            ->join('ekstrakurikuler_master', 'ekstrakurikuler_master.id = ekskul_siswa.ekskul_id')
            ->where('ekskul_siswa.semester', $tahun_ajaran['semester'])
            ->where('ekskul_siswa.tahun_ajaran_id', $tahun_ajaran['id']);
        
        // Menambahkan filter berdasarkan peran user
        if ($is_guru_kelas && $is_penanggung_jawab_ekskul) {
            // Jika berperan ganda, ambil data yang sesuai dengan kelas ATAU ekskul yang dipegang
            $builder->groupStart()
                        ->whereIn('siswa.kelas_id', $guru_kelas_ids)
                        ->orWhereIn('ekskul_siswa.ekskul_id', $ekskul_penanggung_jawab_ids)
                    ->groupEnd();
        } elseif ($is_guru_kelas) {
            // Jika hanya wali kelas, filter berdasarkan kelas
            $builder->whereIn('siswa.kelas_id', $guru_kelas_ids);
        } elseif ($is_penanggung_jawab_ekskul) {
            // Jika hanya penanggung jawab ekskul, filter berdasarkan ekskul
            $builder->whereIn('ekskul_siswa.ekskul_id', $ekskul_penanggung_jawab_ids);
        }

        $siswa_ekskul = $builder->findAll();

        $data = [
            'title' => 'Input Nilai Ekstrakurikuler',
            'siswa_ekskul' => $siswa_ekskul,
        ];

        return view('guru/ekskul_wali_kelas/index', $data);
    }

    public function add()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $ekskul_penanggung_jawab_ids = $this->getPenanggungJawabEkskulIds();
    
        if (empty($guru_kelas_ids) && empty($ekskul_penanggung_jawab_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
            return redirect()->to(base_url('guru/ekskul_wali_kelas'));
        }
    
        $siswa = [];
        $ekskul = [];
    
        // Jika user adalah penanggung jawab ekskul, tampilkan SELURUH siswa
        if (!empty($ekskul_penanggung_jawab_ids)) {
            $siswa = $this->siswaModel
                ->select('siswa.id, siswa.nama_lengkap, kelas.nama_kelas')
                ->join('kelas', 'kelas.id_kelas = siswa.kelas_id')
                ->findAll();
    
            // Tampilkan hanya daftar ekskul yang menjadi tanggung jawabnya
            $ekskul = $this->ekskulMasterModel->whereIn('id', $ekskul_penanggung_jawab_ids)->findAll();
    
        // Jika user HANYA wali kelas (bukan penanggung jawab ekskul), tampilkan siswa di kelasnya saja
        } elseif (!empty($guru_kelas_ids)) {
            $siswa = $this->siswaModel
                ->select('siswa.id, siswa.nama_lengkap, kelas.nama_kelas')
                ->join('kelas', 'kelas.id_kelas = siswa.kelas_id')
                ->whereIn('siswa.kelas_id', $guru_kelas_ids)
                ->findAll();
    
            // Tampilkan semua ekskul (asumsi wali kelas bisa menginput untuk semua ekskul)
            $ekskul = $this->ekskulMasterModel->findAll();
        }
        
        $data = [
            'title' => 'Tambah Ekstrakurikuler',
            'siswa_list' => $siswa,
            'ekskul_list' => $ekskul,
            'validation' => service('validation')
        ];
        
        return view('guru/ekskul_wali_kelas/add', $data);
    }
    
    public function save()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $ekskul_penanggung_jawab_ids = $this->getPenanggungJawabEkskulIds();

        if (empty($guru_kelas_ids) && empty($ekskul_penanggung_jawab_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
            return redirect()->to(base_url('guru/ekskul_wali_kelas'));
        }

        $ekskul_data = $this->request->getVar('ekskul_data');
        $db = \Config\Database::connect();
        $db->transBegin();

        try {
            foreach ($ekskul_data as $id => $data) {
                // Tambahkan validasi izin update untuk setiap data
                $ekskul_siswa = $this->ekskulSiswaModel->find($id);
                if (empty($ekskul_siswa)) continue;

                $siswa = $this->siswaModel->find($ekskul_siswa['siswa_id']);
                $ekskul_master = $this->ekskulMasterModel->find($ekskul_siswa['ekskul_id']);
                
                // Pastikan user adalah wali kelas siswa tersebut ATAU penanggung jawab ekskul tersebut
                if (in_array($siswa['kelas_id'], $guru_kelas_ids) || in_array($ekskul_master['id'], $ekskul_penanggung_jawab_ids)) {
                    $this->ekskulSiswaModel->update($id, ['deskripsi_ekskul' => $data['deskripsi_ekskul']]);
                }
            }
            
            if ($db->transStatus() === false) {
                $db->transRollback();
                session()->setFlashdata('error', 'Gagal menyimpan data ekstrakurikuler.');
            } else {
                $db->transCommit();
                session()->setFlashdata('success', 'Data ekstrakurikuler berhasil disimpan.');
            }
        } catch (\Exception $e) {
            $db->transRollback();
            session()->setFlashdata('error', 'Gagal menyimpan data ekstrakurikuler: ' . $e->getMessage());
        }

        return redirect()->to(base_url('guru/ekskul_wali_kelas'));
    }

    public function save_add()
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $ekskul_penanggung_jawab_ids = $this->getPenanggungJawabEkskulIds();

        if (empty($guru_kelas_ids) && empty($ekskul_penanggung_jawab_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
            return redirect()->to(base_url('guru/ekskul_wali_kelas'));
        }

        $rules = [
            'siswa_id'          => 'required|numeric',
            'ekskul_id'         => 'required|numeric',
            'deskripsi_ekskul'  => 'required'
        ];
        
        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $tahun_ajaran = $this->getActiveTahunAjaran();
        if (empty($tahun_ajaran)) {
            session()->setFlashdata('error', 'Tidak ada tahun ajaran aktif yang ditemukan. Tidak dapat menyimpan.');
            return redirect()->back()->withInput();
        }
        
        $siswa = $this->siswaModel->find($this->request->getVar('siswa_id'));
        $ekskul_master = $this->ekskulMasterModel->find($this->request->getVar('ekskul_id'));

        // Cek izin: user adalah wali kelas siswa ATAU penanggung jawab ekskul
        if (!in_array($siswa['kelas_id'], $guru_kelas_ids) && !in_array($ekskul_master['id'], $ekskul_penanggung_jawab_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk menyimpan data ekskul siswa di kelas tersebut.');
            return redirect()->back()->withInput();
        }
        
        $data = [
            'siswa_id'          => $this->request->getVar('siswa_id'),
            'ekskul_id'         => $this->request->getVar('ekskul_id'),
            'deskripsi_ekskul'  => $this->request->getVar('deskripsi_ekskul'),
            'semester'          => $tahun_ajaran['semester'],
            'tahun_ajaran_id'   => $tahun_ajaran['id'],
        ];

        $this->ekskulSiswaModel->insert($data);

        session()->setFlashdata('success', 'Data ekstrakurikuler berhasil ditambahkan.');
        return redirect()->to(base_url('guru/ekskul_wali_kelas'));
    }

    public function delete($id)
    {
        $guru_kelas_ids = $this->getGuruKelasIds();
        $ekskul_penanggung_jawab_ids = $this->getPenanggungJawabEkskulIds();
        
        if (empty($guru_kelas_ids) && empty($ekskul_penanggung_jawab_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk melakukan aksi ini.');
            return redirect()->to(base_url('guru/ekskul_wali_kelas'));
        }
        
        $ekskul_siswa = $this->ekskulSiswaModel->find($id);
        if (empty($ekskul_siswa)) {
            session()->setFlashdata('error', 'Data tidak ditemukan.');
            return redirect()->to(base_url('guru/ekskul_wali_kelas'));
        }
        
        $siswa = $this->siswaModel->find($ekskul_siswa['siswa_id']);
        $ekskul_master = $this->ekskulMasterModel->find($ekskul_siswa['ekskul_id']);
        
        // Cek izin: user adalah wali kelas siswa ATAU penanggung jawab ekskul
        if (!in_array($siswa['kelas_id'], $guru_kelas_ids) && !in_array($ekskul_master['id'], $ekskul_penanggung_jawab_ids)) {
            session()->setFlashdata('error', 'Anda tidak memiliki izin untuk menghapus data siswa di kelas ini atau ekskul ini.');
            return redirect()->to(base_url('guru/ekskul_wali_kelas'));
        }
        
        $this->ekskulSiswaModel->delete($id);
        session()->setFlashdata('success', 'Data ekstrakurikuler berhasil dihapus.');
        
        return redirect()->to(base_url('guru/ekskul_wali_kelas'));
    }
}