<?php

namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use App\Models\UserModel; // Digunakan untuk berinteraksi dengan tabel users

class Profile extends BaseController
{
    protected $userModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
    }

    /**
     * Menampilkan formulir edit profil dan ganti password.
     */
    public function index()
    {
        $user_id = session()->get('user_id');
        $user_data = $this->userModel->find($user_id); // Ambil data user saat ini

        if (!$user_data) {
            session()->setFlashdata('error', 'Data pengguna tidak ditemukan.');
            return redirect()->to(base_url('guru/dashboard')); // Redirect jika user tidak ditemukan
        }

        $data = [
            'title' => 'Edit Profil & Ganti Password',
            'validation' => service('validation'),
            'user' => session()->get('user'), // Untuk menampilkan info user di layout
            'user_data' => $user_data, // Data user yang akan diedit
        ];
        return view('guru/profile/index', $data);
    }

    /**
     * Memproses permintaan update profil dan/atau ganti password.
     */
    public function update()
{
    $user_id = session()->get('user_id');
    $user = $this->userModel->find($user_id);

    if (!$user) {
        session()->setFlashdata('error', 'Pengguna tidak ditemukan.');
        return redirect()->back();
    }

    $validationRules = [];
    $data_to_update = [];

    // Validasi dan siapkan data untuk NAMA
    if ($this->request->getPost('nama') !== $user['nama']) {
        $validationRules['nama'] = 'required|min_length[3]';
        $data_to_update['nama'] = $this->request->getPost('nama');
    }

    // Validasi dan siapkan data untuk NIP
    // Hapus aturan 'required' agar NIP bisa kosong atau diisi '-'
    if ($this->request->getPost('nip') !== $user['nip']) {
        $validationRules['nip'] = 'permit_empty|min_length[1]';
        $data_to_update['nip'] = $this->request->getPost('nip');
    }

    // Validasi dan siapkan data untuk USERNAME
    if ($this->request->getPost('username') !== $user['username']) {
        $validationRules['username'] = "required|min_length[3]|is_unique[users.username,id,{$user_id}]";
        $data_to_update['username'] = $this->request->getPost('username');
    }

    // Validasi dan siapkan data untuk PASSWORD
    $old_password_input = $this->request->getPost('old_password');
    $new_password_input = $this->request->getPost('new_password');

    if (!empty($new_password_input)) {
        $validationRules['new_password'] = 'required|min_length[6]';
        $validationRules['confirm_password'] = 'required|matches[new_password]';

        if (!$old_password_input || !password_verify($old_password_input, $user['password'])) {
            session()->setFlashdata('error', 'Password lama salah.');
            return redirect()->back()->withInput();
        }
        $data_to_update['password'] = $new_password_input;
    }

    // Cek jika tidak ada data yang diubah
    if (empty($data_to_update)) {
        session()->setFlashdata('success', 'Tidak ada perubahan data yang disimpan.');
        return redirect()->to(base_url('guru/profile'));
    }

    // Lakukan validasi hanya pada data yang akan di-update
    if (!$this->validate($validationRules)) {
        session()->setFlashdata('errors', $this->validator->getErrors());
        return redirect()->back()->withInput();
    }

    // --- Update Data User di Database ---
    try {
        $this->userModel->update($user_id, $data_to_update);
        
        // Perbarui data sesi (hanya jika ada perubahan)
        $user_session_data = session()->get('user');
        if (isset($data_to_update['nama'])) {
            $user_session_data['nama'] = $data_to_update['nama'];
        }
        if (isset($data_to_update['nip'])) {
            $user_session_data['nip'] = $data_to_update['nip'];
        }
        if (isset($data_to_update['username'])) {
            $user_session_data['username'] = $data_to_update['username'];
        }
        session()->set('user', $user_session_data);

        session()->setFlashdata('success', 'Profil dan/atau password berhasil diperbarui.');
        return redirect()->to(base_url('guru/profile'));

    } catch (\Exception $e) {
        log_message('error', 'Gagal memperbarui profil: ' . $e->getMessage());
        session()->setFlashdata('error', 'Terjadi kesalahan saat menyimpan data. Silakan coba lagi.');
        return redirect()->back()->withInput();
    }
}
}